/**
 * Copyright (C) 2018 Achim Kaiser
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.kaisersite.mylibrary;

import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.IBinder;
import androidx.annotation.Nullable;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import android.support.v4.media.MediaMetadataCompat;
import androidx.core.app.NotificationCompat;
import androidx.media.session.MediaButtonReceiver;
import android.support.v4.media.session.MediaSessionCompat;
import android.support.v4.media.session.PlaybackStateCompat;
import android.util.Log;
import androidx.media.app.NotificationCompat.MediaStyle;

public class MyMediaService extends Service
{
    private static final String TAG = "MyMediaService";
    private String playingid;
    private String playlistid;
    MediaSessionCompat mSession;
    private PlaybackStateCompat.Builder mPlaybackStateBuilder;
    private MediaMetadataCompat.Builder mMediaBuilder;
    private MediaNotificationHolder medianotification;
    private DeviceNotificationHolder devicenotification;
    private IPlayer mMediaPlayer;
    public static final String UPDATE_PLAYING = "update_playing";
    public static final String CLOSE_PLAYER_CLOSE = "close_player_close";
    public static final String CLOSE_PLAYER = "close_player";
    public static final String CLOSE_PLAYER_STOP_CLOSE = "close_player_stop_close";
    public static final String REQUEST_PLAYSTATUS = "request_playstatus";
    public static final String SEEKTO = "seekto";
    public static final String UPDATE_NEXT = "update_next";
    private NotificationManager mNotificationManager;
    private boolean bLocal=false;

    public MyMediaService()
    {
    }

    private class MediaNotificationHolder
    {
        private boolean isPlaying=true;
        private String title;
        private String description;
        private Bitmap bitmap;

        public void setdata(String title, String description, Bitmap b)
        {
            this.title=title;
            this.description=description;
            this.bitmap=b;
        }

        private NotificationCompat.Builder createNotification()
        {
            MediaStyle style = new MediaStyle().setMediaSession(mSession.getSessionToken()).setShowActionsInCompactView(0,1,2);

            Intent notificationIntent = Tools.getMainIntent(MyMediaService.this);
            notificationIntent.setAction(Intent.ACTION_MAIN);
            notificationIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
            PendingIntent pendingIntent = PendingIntent.getActivity(MyMediaService.this, 0,
                    notificationIntent, 0);

            Intent stopIntent = new Intent(CLOSE_PLAYER);
            PendingIntent stopPendingIntent = PendingIntent.getBroadcast(MyMediaService.this, 0,
                    stopIntent, 0);

            NotificationCompat.Builder notification = new NotificationCompat.Builder(MyMediaService.this, "media")
                    .setSmallIcon(R.drawable.ic_notification)
                    .setDeleteIntent(stopPendingIntent)
                    .setContentIntent(pendingIntent)
                    .setOnlyAlertOnce(true)
                    .setPriority(NotificationCompat.PRIORITY_LOW)
                    .setStyle(style)
                    .setContentTitle(title)
                    .setContentText(description)
                    .setLargeIcon(bitmap);
            notification.addAction(new NotificationCompat.Action(
                    android.R.drawable.ic_media_previous, getString(de.kaisersite.mylibrary.R.string.app_name),
                    MediaButtonReceiver.buildMediaButtonPendingIntent(MyMediaService.this,
                            PlaybackStateCompat.ACTION_SKIP_TO_PREVIOUS)));
            if (isPlaying)
                notification.addAction(new NotificationCompat.Action(
                        android.R.drawable.ic_media_pause, getString(de.kaisersite.mylibrary.R.string.app_name),
                        MediaButtonReceiver.buildMediaButtonPendingIntent(MyMediaService.this,
                                PlaybackStateCompat.ACTION_PAUSE)));
            else
                notification.addAction(new NotificationCompat.Action(
                        android.R.drawable.ic_media_play, getString(de.kaisersite.mylibrary.R.string.app_name),
                        MediaButtonReceiver.buildMediaButtonPendingIntent(MyMediaService.this,
                                PlaybackStateCompat.ACTION_PLAY)));
            notification.addAction(new NotificationCompat.Action(
                    android.R.drawable.ic_media_next, getString(de.kaisersite.mylibrary.R.string.app_name),
                    MediaButtonReceiver.buildMediaButtonPendingIntent(MyMediaService.this,
                            PlaybackStateCompat.ACTION_SKIP_TO_NEXT)));
            notification.setSmallIcon(R.drawable.ic_notification);

            return notification;
        }

        public void setIsPlaying(boolean isPlaying)
        {
            this.isPlaying = isPlaying;
        }
    }

    private class DeviceNotificationHolder
    {
        private String device;
        private String status;
        private boolean bOngoing=true;

        public void setdata(String device, String status)
        {
            this.device=device;
            this.status=status;
        }

        public void setOngoing(boolean ongoing)
        {
            this.bOngoing=ongoing;
        }


        private NotificationCompat.Builder createNotification()
        {
            Intent notificationIntent = Tools.getMainIntent(MyMediaService.this);
            notificationIntent.setAction(Intent.ACTION_MAIN);
            notificationIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
            PendingIntent pendingIntent = PendingIntent.getActivity(MyMediaService.this, 0,
                    notificationIntent, 0);

            Intent stopIntent = new Intent(CLOSE_PLAYER);
            PendingIntent stopPendingIntent = PendingIntent.getBroadcast(MyMediaService.this, 0,
                    stopIntent, 0);

            NotificationCompat.Builder notification = new NotificationCompat.Builder(MyMediaService.this, "media")
                    .setSmallIcon(R.drawable.ic_notification)
                    .setDeleteIntent(stopPendingIntent)
                    //.setContentIntent(pendingIntent)
                    .setOnlyAlertOnce(true)
                    .setOngoing(bOngoing)
                    //.setPriority(NotificationCompat.PRIORITY_LOW)
                    .setContentTitle(device);
            if (status!=null)
                notification.setContentText(status);
            if (bOngoing)
                {
                if (device==null || device.equals("local"))
                    {
                    notification.addAction(0, getString(R.string.open), pendingIntent);

                    stopIntent = new Intent(CLOSE_PLAYER_CLOSE);
                    stopPendingIntent = PendingIntent.getBroadcast(MyMediaService.this, 1,
                            stopIntent, 0);
                    notification.addAction(0, getString(R.string.action_close), stopPendingIntent);
                    }
                else
                    {
                    notification.addAction(0, getString(R.string.open), pendingIntent);

                    stopIntent = new Intent(CLOSE_PLAYER_CLOSE);
                    stopPendingIntent = PendingIntent.getBroadcast(MyMediaService.this, 1,
                            stopIntent, 0);
                    notification.addAction(0, getString(R.string.action_close), stopPendingIntent);
                    stopIntent = new Intent(CLOSE_PLAYER_STOP_CLOSE);
                    stopPendingIntent = PendingIntent.getBroadcast(MyMediaService.this, 1,
                            stopIntent, 0);
                    notification.addAction(0, getString(R.string.action_stop_close), stopPendingIntent);
                    }
                }
            return notification;
        }
    }

    @Override
    public void onCreate()
    {
        super.onCreate();

        IntentFilter filter = new IntentFilter();
        filter.addAction(CLOSE_PLAYER);
        filter.addAction(CLOSE_PLAYER_CLOSE);
        filter.addAction(CLOSE_PLAYER_STOP_CLOSE);
        registerReceiver(mMessageReceiverClosePlayer, filter);
        LocalBroadcastManager.getInstance(this).registerReceiver(mMessageReceiverRequestPlayStatus, new IntentFilter(REQUEST_PLAYSTATUS));
        LocalBroadcastManager.getInstance(this).registerReceiver(mMessageReceiverSeekTo, new IntentFilter(SEEKTO));
        LocalBroadcastManager.getInstance(this).registerReceiver(mMessageReceiverUpdateNext, new IntentFilter(UPDATE_NEXT));
        LocalBroadcastManager.getInstance(this).registerReceiver(mMessageReceiverCastInfo, new IntentFilter(MyMediaPlayerUPNP.UPNP_INFO));

        // Start a new MediaSession
        createMediaSession();

        createMediaMetadataBuilder();

        createPlayBackStateBuilder();

        createNotificationChannel();
        medianotification =new MediaNotificationHolder();
        if (getResources().getBoolean(R.bool.wear))
            devicenotification =new DeviceNotificationHolder();
        startForeground(1, medianotification.createNotification().build());
    }

    private IPlayer createMediaPlayer(String deviceid, String devicename)
    {
        IPlayer iPlayer;
        if (devicenotification != null)
            {
            if (devicename==null)
                bLocal=true;
            devicenotification.setdata(devicename==null?"local":devicename , null);
            mNotificationManager.notify(2, devicenotification.createNotification().build());
            }
        if (deviceid==null)
            {
            iPlayer = new MyMediaPlayerDual();
            }
        else
            {
            iPlayer = new MyMediaPlayerUPNP(this);
            }
        iPlayer.setOnPlayInfoListener(mOnPlayInfoListener);
        return iPlayer;
    }

    private void createPlayBackStateBuilder()
    {
        mPlaybackStateBuilder = new PlaybackStateCompat.Builder()
                .setState(PlaybackStateCompat.STATE_STOPPED, 0, 1)
                .setActions(PlaybackStateCompat.ACTION_SKIP_TO_NEXT
                        | PlaybackStateCompat.ACTION_SKIP_TO_PREVIOUS
                        | PlaybackStateCompat.ACTION_PLAY);
    }


    private void createMediaMetadataBuilder()
    {
        mMediaBuilder = new MediaMetadataCompat.Builder();
    }

    private void createMediaSession()
    {
        mSession = new MediaSessionCompat(this, "MusicService");
        mSession.setActive(true);
        mSession.setCallback(new MediaSessionCallback());
        mSession.setFlags(MediaSessionCompat.FLAG_HANDLES_MEDIA_BUTTONS |
                MediaSessionCompat.FLAG_HANDLES_TRANSPORT_CONTROLS);
    }

    @Override
    public void onDestroy()
    {
        mNotificationManager.cancel(1);
        if (bLocal)
            {
            mNotificationManager.cancel(2);
            }
        else if (devicenotification!=null)
            {
            devicenotification.setOngoing(false);
            mNotificationManager.notify(2, devicenotification.createNotification().build());
            }
        unregisterReceiver(mMessageReceiverClosePlayer);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mMessageReceiverRequestPlayStatus);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mMessageReceiverSeekTo);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mMessageReceiverUpdateNext);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mMessageReceiverCastInfo);
        mSession.setActive(false);
        mSession.release();
        if (mMediaPlayer!=null)
            mMediaPlayer.release();

        // Empty possible playlist
        sendPlayInfo(null, null, true, 0,0);

        Log.d(TAG, "onDestroy: MediaPlayerAdapter stopped, and MediaSession released");
    }

    private void createNotificationChannel()
    {
        mNotificationManager = (NotificationManager) getSystemService(Context.NOTIFICATION_SERVICE);
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O)
            {
            NotificationManager notificationManager = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);
            NotificationChannel notificationChannel = new NotificationChannel("media", "AKWearPlayer", NotificationManager.IMPORTANCE_DEFAULT);
            notificationChannel.enableVibration(false);
            notificationChannel.setImportance(NotificationManager.IMPORTANCE_LOW);
            mNotificationManager.createNotificationChannel(notificationChannel);
            }
    }


    @Override
    public int onStartCommand(Intent intent, int flags, int startId)
    {
        if (MediaButtonReceiver.handleIntent(mSession, intent)!=null)
            return super.onStartCommand(intent,flags,startId);
        if (intent==null)
            return START_STICKY;
        Log.d(TAG, "onStartCommand");
        String id = intent.getStringExtra("id");
        String pid = intent.getStringExtra("playlistid");
        if (mMediaPlayer==null)
            mMediaPlayer=createMediaPlayer(CurrentDevice.getId(this),CurrentDevice.getName(this));
        if (mMediaPlayer.isPlaying() && id.equals(playingid) && pid.equals(playlistid) )
            {
            pausePlaying();
            }
        else
            {
            startPlaying(id,pid);
            }

        return START_STICKY;
    }

    private void startPlaying(String id,String pid)
    {
        playlistid=pid;
        PlaylistDB pldb = new PlaylistDB(this, pid);
        while (!mMediaPlayer.start(pldb.getItem(id)))
            {
            pldb.setState(id, PlayListModel.STATE_ERROR);
            sendUpdateBroadcast(id);
            id=pldb.getNextId(id);
            if (id==null)
                {
                stopSelf();
                return;
                }
            }
        setInfo(pldb.getItem(id));
    }

    private void setInfo(PlayListModel plm)
    {
        playingid=plm.getId();
        Bitmap b = Tools.getMediaBitmap(plm.getFile());
        if (b==null)
            b=BitmapFactory.decodeResource(getResources(), de.kaisersite.mylibrary.R.drawable.ic_file);
        mMediaBuilder.putText(MediaMetadataCompat.METADATA_KEY_TITLE, plm.getTitle())
                .putText(MediaMetadataCompat.METADATA_KEY_ARTIST, plm.getDescription())
                .putBitmap(MediaMetadataCompat.METADATA_KEY_ART, b);
        mSession.setMetadata(mMediaBuilder.build());
        medianotification.setdata(plm.getTitle(),plm.getDescription(),b);
        mNotificationManager.notify(1, medianotification.createNotification().build());
    }


    private void pausePlaying()
    {
        mMediaPlayer.pause();
    }

    private void sendPlayInfo(String id, String pid, boolean paused, int currentPosition, int duration)
    {

        if (id!=null)
            {
            if (paused)
                {
                mPlaybackStateBuilder.setState(PlaybackStateCompat.STATE_PAUSED, PlaybackStateCompat.PLAYBACK_POSITION_UNKNOWN, 1)
                        .setActions(PlaybackStateCompat.ACTION_SKIP_TO_NEXT
                                | PlaybackStateCompat.ACTION_SKIP_TO_PREVIOUS
                                | PlaybackStateCompat.ACTION_STOP
                                | PlaybackStateCompat.ACTION_PLAY
                                | PlaybackStateCompat.ACTION_SEEK_TO);
                mSession.setPlaybackState(mPlaybackStateBuilder.build());
                medianotification.setIsPlaying(false);
                mNotificationManager.notify(1, medianotification.createNotification().build());
                stopForeground(false);
                }
            else
                {
                mPlaybackStateBuilder.setState(PlaybackStateCompat.STATE_PLAYING, 0, 1)
                        .setActions(PlaybackStateCompat.ACTION_SKIP_TO_NEXT
                                | PlaybackStateCompat.ACTION_SKIP_TO_PREVIOUS
                                | PlaybackStateCompat.ACTION_STOP
                                | PlaybackStateCompat.ACTION_PAUSE
                                | PlaybackStateCompat.ACTION_SEEK_TO);
                mSession.setPlaybackState(mPlaybackStateBuilder.build());
                medianotification.setIsPlaying(true);
                startForeground(1, medianotification.createNotification().build());
                }
            }

        Intent bcintent = new Intent(UPDATE_PLAYING);
        //put whatever data you want to send, if any
        if (id != null)
            bcintent.putExtra("id", id);
        if (pid != null)
            bcintent.putExtra("playlistid", pid);
        bcintent.putExtra("paused", paused);
        bcintent.putExtra("currentpos",currentPosition);
        bcintent.putExtra("duration",duration);
        LocalBroadcastManager.getInstance(this).sendBroadcast(bcintent);
    }


    @Nullable
    @Override
    public IBinder onBind(Intent intent)
    {
        return null;
    }

    private BroadcastReceiver mMessageReceiverClosePlayer = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            if (intent.getAction().equals(CLOSE_PLAYER_CLOSE) || intent.getAction().equals(CLOSE_PLAYER_STOP_CLOSE))
                {
                CurrentDevice.setDevice(MyMediaService.this, null, null);
                mNotificationManager.cancel(2);
                devicenotification=null;
                }
            // stop hält auch den remoteplayer an
            if (intent.getAction().equals(CLOSE_PLAYER_STOP_CLOSE))
                mMediaPlayer.stop();
            else
                stopSelf();
        }
    };

    private BroadcastReceiver mMessageReceiverRequestPlayStatus = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            sendPlayInfo(playingid,playlistid, !mMediaPlayer.isPlaying(), mMediaPlayer.getCurrentPosition(), mMediaPlayer.getDuration());
        }
    };

    private BroadcastReceiver mMessageReceiverSeekTo = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            mMediaPlayer.seekTo(intent.getIntExtra("seekto",0));
            sendPlayInfo(playingid,playlistid, !mMediaPlayer.isPlaying(), mMediaPlayer.getCurrentPosition(), mMediaPlayer.getDuration());
        }
    };

    private BroadcastReceiver mMessageReceiverUpdateNext = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            PlaylistDB pldb = new PlaylistDB(MyMediaService.this,playlistid);
            String newid = pldb.getNextId(playingid);
            if (newid != null)
                {
                mMediaPlayer.setNext(pldb.getItem(newid));
                }
            else
                {
                mMediaPlayer.setNext(null);
                }
        }
    };

    private IPlayer.OnPlayInfoListener mOnPlayInfoListener = mOnPlayInfoListener = new IPlayer.OnPlayInfoListener()
    {
        @Override
        public void onPlayInfo(PlayListModel plm)
        {
            PlaylistDB pldb = new PlaylistDB(MyMediaService.this,playlistid);
            if (plm!=null)
                {
                if (!plm.getId().equals(playingid))
                    {
                    setInfo(plm);
                    }
                sendPlayInfo(playingid, playlistid, !mMediaPlayer.isPlaying(), mMediaPlayer.getCurrentPosition(),mMediaPlayer.getDuration());
                String newid = pldb.getNextId(plm.getId());
                if (newid != null)
                    {
                    // set new next
                    mMediaPlayer.setNext(pldb.getItem(newid));
                    }
                else
                    {
                    mMediaPlayer.setNext(null);
                    }
                }
            else
                {
                String newid = pldb.getNextId(playingid);
                if (newid != null)
                    {
                    startPlaying(newid,playlistid);
                    }
                else
                    stopSelf();
                }
        }
    };

    private class MediaSessionCallback extends MediaSessionCompat.Callback
    {
        @Override
        public void onPlay()
        {
            Log.d(TAG, "play");
            startPlaying(playingid, playlistid);
        }

        @Override
        public void onSkipToQueueItem(long queueId)
        {
            Log.d(TAG, "OnSkipToQueueItem:" + queueId);
        }

        @Override
        public void onSeekTo(long position)
        {
            Log.d(TAG, "onSeekTo:" + position);
        }

        @Override
        public void onPause()
        {
            Log.d(TAG, "pause.");
            pausePlaying();
        }

        @Override
        public void onStop()
        {
            Log.d(TAG, "stop. ");
            stopSelf();
        }

        @Override
        public void onSkipToNext()
        {
            Log.d(TAG, "skipToNext");
            PlaylistDB pldb = new PlaylistDB(MyMediaService.this,playlistid);
            String newid = pldb.getNextId(playingid);
            if (newid != null)
                {
                if (!mMediaPlayer.isPlaying())
                    {
                    startForeground(1, medianotification.createNotification().build());
                    setInfo(pldb.getItem(newid));
                    stopForeground(false);
                    }
                else
                    startPlaying(newid,playlistid);
                }
        }

        @Override
        public void onSkipToPrevious()
        {
            Log.d(TAG, "onSkipToPrevious");
            PlaylistDB pldb = new PlaylistDB(MyMediaService.this,playlistid);
            String newid = pldb.getPrevId(playingid);
            if (mMediaPlayer.isPlaying() && newid == null)
                {
                mMediaPlayer.seekTo(0);
                sendPlayInfo(playingid,playlistid, !mMediaPlayer.isPlaying(), mMediaPlayer.getCurrentPosition(), mMediaPlayer.getDuration());
                return;
                }
            if (newid != null)
                {
                if (!mMediaPlayer.isPlaying())
                    {
                    startForeground(1, medianotification.createNotification().build());
                    setInfo(pldb.getItem(newid));
                    stopForeground(false);
                    }
                else
                    startPlaying(newid,playlistid);
                }
        }

    }

    private void sendUpdateBroadcast( String id)
    {
        Intent bcintent = new Intent(DownloadJobService.UPDATE_DOWNLOAD);

        if (id!=null)
        //put whatever data you want to send, if any
            bcintent.putExtra("id", id);

        //send broadcast
        LocalBroadcastManager.getInstance(this).sendBroadcast(bcintent);
    }

    private BroadcastReceiver mMessageReceiverCastInfo = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            Log.d(TAG, "receive");
            if (devicenotification != null)
                {
                devicenotification.setdata(CurrentDevice.getName(MyMediaService.this), intent.getStringExtra("statustext"));
                mNotificationManager.notify(2, devicenotification.createNotification().build());
                }
        }
    };

}
