/**
 * Copyright (C) 2014 Casper Jørgensen
 * Modifications copyright (C) 2018 Achim Kaiser
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.kaisersite.mylibrary;

import android.content.res.Resources;
import android.net.Uri;

import org.fourthline.cling.model.meta.Service;
import org.fourthline.cling.support.model.DIDLObject;
import org.fourthline.cling.support.model.Res;
import org.fourthline.cling.support.model.container.Container;
import org.fourthline.cling.support.model.item.Item;

import java.util.List;

import de.kaisersite.mylibrary.CustomListItem;

public class ItemModel extends CustomListItem
{
    private final Resources res;
    private final Service service;
    private final DIDLObject item;

    public ItemModel(Resources res, int icon, Service service, DIDLObject item)
    {
        super(icon);

        this.res = res;
        this.service = service;
        this.item = item;
    }

    public String getUrl()
    {
        if (item == null)
            return "N/A";

        Res resource = item.getFirstResource();

        if (resource == null || resource.getValue() == null)
            return "N/A";

        return resource.getValue();
    }

    public long getDuration()
    {
        if (item == null)
            return -1;

        Res resource = item.getFirstResource();

        if (resource == null || resource.getDuration() == null)
            return -1;

        long l=-1;
        try
            {
            String[] durarr=resource.getDuration().split(":");

            l=((Long.parseLong(durarr[0]) * 60 + Long.parseLong(durarr[1])) * 60 + Long.parseLong(durarr[2])) * 1000;
            }
        catch (Exception e)
            {

            }
        return l;
    }

    public Item getItem()
    {
        if (isContainer())
            return null;

        return (Item) item;
    }

    public Container getContainer()
    {
        if (!isContainer())
            return null;

        return (Container) item;
    }

    public Service getService()
    {
        return this.service;
    }

    public boolean isContainer()
    {
        return item instanceof Container;
    }

    @Override
    public String getId()
    {
        return item.getId();
    }

    @Override
    public String getTitle()
    {
        return item.getTitle();
    }

    @Override
    public String getDescription()
    {
        if (isContainer())
            {
            Integer children = getContainer().getChildCount();

            if (children != null)
                return getContainer().getChildCount() + " " + res.getString(R.string.infoitems_items);

            return res.getString(R.string.infoitems_folder);
            }

        List<DIDLObject.Property> properties = item.getProperties();
        if (properties != null && properties.size() != 0)
            {
            for (DIDLObject.Property property : properties)
                {
                if (property.getDescriptorName().equalsIgnoreCase("artist"))
                    return property.getValue().toString();
                }
            for (DIDLObject.Property property : properties)
                {
                if (property.getDescriptorName().equalsIgnoreCase("date"))
                    return property.getValue().toString();
                }
            }

        List<Res> resources = item.getResources();
        if (resources != null && resources.size() != 0)
            {

            Res resource = item.getResources().get(0);
            String resolution = resource.getResolution();

            if (resolution != null)
                return resolution;

            String creator = item.getCreator();
            if (creator == null)
                return res.getString(R.string.infoitems_file);

            if (creator.startsWith("Unknown"))
                return null;

            return creator;
            }

        return "N/A";
    }

    @Override
    public String getDescription2()
    {
        if (!isContainer())
            {
            List<DIDLObject.Property> properties = item.getProperties();
            if (properties != null && properties.size() != 0)
                {
                for (DIDLObject.Property property : properties)
                    {
                    if (property.getDescriptorName().equalsIgnoreCase("album"))
                        return property.getValue().toString();
                    }
                }

            Uri uri = Uri.parse(getUrl());
            MimeTypeMap mime = MimeTypeMap.getSingleton();
            return mime.getMimeTypeFromUrl(uri.toString());
            }

        String genre = item.getFirstPropertyValue(DIDLObject.Property.UPNP.GENRE.class);
        if (genre == null)
            return null;

        if (genre.startsWith("Unknown"))
            return null;

        return genre;
    }
}
