/**
 * Copyright (C) 2018 Achim Kaiser
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.kaisersite.mylibrary;

import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.SystemClock;
import androidx.core.app.JobIntentService;
import androidx.core.app.NotificationCompat;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import android.util.Log;
import android.widget.Toast;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;


public class DownloadJobService extends JobIntentService
{
    public static final String UPDATE_DOWNLOAD = "update_download";

    public static final int JOB_ID = 111;

    public static void enqueueWork(Context context, Intent work)
    {
        enqueueWork(context, DownloadJobService.class, JOB_ID, work);
    }

    @Override
    public void onCreate()
    {
        super.onCreate();
        NetworkManaging.getInstance(this).addRegistration();
    }

    @Override
    protected void onHandleWork(Intent intent)
    {
        boolean internetavailable = NetworkManaging.getInstance(this).waitForInternet();
        int status = PlayListModel.STATE_LOADING;
        if (!isStopped() && internetavailable)
            {
            Log.i("DownloadJobService", "Executing work: " + intent);

            Intent notificationIntent = Tools.getMainIntent(this);
            notificationIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
            PendingIntent pendingIntent = PendingIntent.getActivity(this, 0,
                    notificationIntent, 0);

            Intent cancelIntent = Tools.getMainIntent(this);
            cancelIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
            cancelIntent.putExtra("cancel", true);
            PendingIntent cancelPendingIntent =
                    PendingIntent.getActivity(this, 1, cancelIntent, 0);

            NotificationCompat.Action.WearableExtender actionExtender =
                    new NotificationCompat.Action.WearableExtender()
                            .setHintLaunchesActivity(true)
                            .setHintDisplayActionInline(true);
            NotificationCompat.WearableExtender wearableExtender =
                    new NotificationCompat.WearableExtender()
                            .setHintHideIcon(true);
            wearableExtender.addAction(new NotificationCompat.Action.Builder(android.R.drawable.ic_menu_close_clear_cancel, getString(R.string.action_close), cancelPendingIntent).extend(actionExtender).build());

            Bitmap icon = BitmapFactory.decodeResource(getResources(), R.drawable.ic_file);


            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O)
                {
                NotificationManager notificationManager = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);
                NotificationChannel notificationChannel = new NotificationChannel("download", "Download", NotificationManager.IMPORTANCE_DEFAULT);
                notificationChannel.setImportance(NotificationManager.IMPORTANCE_LOW);
                notificationChannel.enableVibration(false);
                notificationManager.createNotificationChannel(notificationChannel);
                }

            NotificationCompat.Builder notification = new NotificationCompat.Builder(this, "download")
                    .setContentTitle(intent.getStringExtra("title"))
                    .setContentText(intent.getStringExtra("desc"))
                    .setSmallIcon(R.mipmap.ic_launcher)
                    .addAction(android.R.drawable.ic_menu_close_clear_cancel, getString(R.string.action_close),cancelPendingIntent)
                    .setPriority(NotificationCompat.PRIORITY_LOW)
                    .setLargeIcon(
                            Bitmap.createScaledBitmap(icon, 128, 128, false))
                    .setContentIntent(pendingIntent)
                    .setProgress(0, 0, true)
                    .setOnlyAlertOnce(true)
                    .setOngoing(true)
                    .extend(wearableExtender);

            startForeground(111, notification.build());

            HttpURLConnection c = null;
            InputStream in = null;
            FileOutputStream f = null;
            try
                {
                new File(intent.getStringExtra("file")).delete();
                URL u = new URL(intent.getStringExtra("url"));
                c = (HttpURLConnection) u.openConnection();
                c.setRequestMethod("GET");
                //c.setDoOutput(true);
                c.connect();

                int lenghtOfFile = c.getContentLength();

                f = new FileOutputStream(intent.getStringExtra("file"));

                in = c.getInputStream();
                long systemtime = System.currentTimeMillis();
                byte[] buffer = new byte[4096];
                int len1 = 0;
                long total = 0;
                while ((len1 = in.read(buffer)) > 0 && !isStopped())
                    {
                    total += len1;
                    f.write(buffer, 0, len1);
                    // This doesn't work because the notification is brought to front every time... why?
                    if (!getResources().getBoolean(R.bool.wear) && systemtime+500<System.currentTimeMillis())
                        {
                        systemtime=System.currentTimeMillis();
                        notification.setProgress(100, (int) ((total * 100) / lenghtOfFile), false);
                        startForeground(111, notification.build());
                        }
                    }
                status = PlayListModel.STATE_LOADED;
                }
            catch (ProtocolException e)
                {
                status = PlayListModel.STATE_ERROR;
                e.printStackTrace();
                }
            catch (MalformedURLException e)
                {
                status = PlayListModel.STATE_ERROR;
                e.printStackTrace();
                }
            catch (IOException e)
                {
                status = PlayListModel.STATE_ERROR;
                e.printStackTrace();
                }
            try
                {
                if (f != null)
                    f.close();
                if (in != null)
                    in.close();
                if (c != null)
                    c.disconnect();
                }
            catch (IOException e)
                {
                e.printStackTrace();
                status = PlayListModel.STATE_ERROR;
                }
            }

        if (isStopped() || !internetavailable)
            {
            status = PlayListModel.STATE_ERROR;
            new File(intent.getStringExtra("file")).delete();
            }

        new PlaylistDB(this,intent.getStringExtra("playlistid")).setState(intent.getStringExtra("id"), status);

        sendUpdateBroadcast(this,intent.getStringExtra("id"));
        // Falls ein neues lied kommt info darüber an den Player
        Intent bcintent = new Intent(MyMediaService.UPDATE_NEXT);
        LocalBroadcastManager.getInstance(this).sendBroadcast(bcintent);

        Log.i("DownloadJobService", "Completed service @ " + SystemClock.elapsedRealtime());
    }

    @Override
    public void onDestroy()
    {
        super.onDestroy();
        NotificationManager notificationManager = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);
        notificationManager.cancel(111);
        PlaylistDB.setAllLoadingToError(this);
        sendUpdateBroadcast(this,null);
        NetworkManaging.getInstance(getApplicationContext()).removeRegistration();
        //toast("All work complete");
    }

    @Override
    public boolean onStopCurrentWork()
    {
        return false;
    }

    public static void startDownload(final Context context, String title, String description, String url,String description2, long duration)
    {
        String pid=PlaylistDB.getCurrentPlaylistId(context);
        String sId=new PlaylistDB(context,pid).existsItem(title,description);
         if (sId!=null)
            {
            new PlaylistDB(context,pid).moveToLast(sId);
            //Tools.showToast(context,"exists");
            sendUpdateBroadcast(context,null);
            return;
            }
        Intent intent = new Intent();
        intent.putExtra("title", title);
        intent.putExtra("desc", description);
        intent.putExtra("url", url);
        //File[] file=context.getExternalFilesDirs(Environment.DIRECTORY_MUSIC);
        File[] file = context.getExternalMediaDirs();
        // find correct directory
        int i=file.length-1;
        while (i!=0 || file[i]==null || !file[i].isDirectory() || !file[i].canWrite())
            i=i-1;
        int id = 0;
        while (!new PlaylistDB(context,pid).addItem(url, "" + id, new File(file[i], pid + "-" + id + ".mp3").getPath(), title, description,description2,duration))
            id++;
        sendUpdateBroadcast(context,null);
        intent.putExtra("file", new File(file[i], pid + "-" + id + ".mp3").getPath());
        intent.putExtra("id", "" + id);
        intent.putExtra("playlistid", "" + pid);
        DownloadJobService.enqueueWork(context, intent);
        Toast.makeText(context,R.string.download,Toast.LENGTH_SHORT).show();
        /*toast.setText(R.string.download);
        toast.setDuration(Toast.LENGTH_SHORT);
        toast.show();*/
    }

    private static void sendUpdateBroadcast(Context context, String id)
    {
        Intent bcintent = new Intent(UPDATE_DOWNLOAD);

        if (id!=null)
            //put whatever data you want to send, if any
            bcintent.putExtra("id", id);

        //send broadcast
        LocalBroadcastManager.getInstance(context).sendBroadcast(bcintent);
    }

}
