package de.kaisersite.akwearplayer;

import android.app.FragmentManager;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.provider.Settings;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import androidx.swiperefreshlayout.widget.SwipeRefreshLayout;
import androidx.appcompat.app.ActionBar;
import androidx.appcompat.app.AppCompatActivity;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import android.util.Log;
import android.view.View;
import android.widget.LinearLayout;

import java.util.ArrayList;

import de.kaisersite.mylibrary.ContentDirectoryBrowseTaskFragment;
import de.kaisersite.mylibrary.CustomListItem;
import de.kaisersite.mylibrary.DeviceModel;
import de.kaisersite.mylibrary.ItemModel;
import de.kaisersite.mylibrary.MyUPNPListAdapter;
import de.kaisersite.mylibrary.NetworkManaging;

public class UPNPActivity extends AppCompatActivity implements ContentDirectoryBrowseTaskFragment.Callbacks, MyUPNPListAdapter.Controller, SwipeRefreshLayout.OnRefreshListener
{
    private static final String TAG = "UPNPActivity";
    private RecyclerView mRecyclerView;

    private ContentDirectoryBrowseTaskFragment mFragment;
    private ArrayList<CustomListItem> mDeviceList;
    private MyUPNPListAdapter mDeviceListAdapter;
    private ArrayList<CustomListItem> mItemList;
    private MyUPNPListAdapter mItemListAdapter;
    private SwipeRefreshLayout mSwipeToRefreshLayout;
    private LinearLayout mWaitForWLAN;
    private LinearLayout mWaitForDevices;
    private MyUPNPListAdapter mCurrentAdapter;
    private boolean bBrowseAVTransport=false;

    @Override
    protected void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        bBrowseAVTransport = getIntent().getBooleanExtra("BrowseAVTransport", false);

        setContentView(R.layout.activity_upnp);
        mWaitForWLAN = findViewById(R.id.wait_for_wifi);
        mWaitForDevices = findViewById(R.id.wait_for_devices);

        mRecyclerView = findViewById(R.id.recycler_launcher_view);
        mRecyclerView.setLayoutManager(
                new LinearLayoutManager(this));

        mSwipeToRefreshLayout = findViewById(R.id.swiperefresh);
        mSwipeToRefreshLayout.setOnRefreshListener(this);

        mDeviceList=new ArrayList<>();
        mDeviceListAdapter = new MyUPNPListAdapter(this, this, mDeviceList);
        mDeviceListAdapter.setHasStableIds(true);
        mItemList=new ArrayList<>();
        mItemListAdapter = new MyUPNPListAdapter(this, this, mItemList);
        mItemListAdapter.setHasStableIds(true);
        mRecyclerView.setAdapter(mDeviceListAdapter);
        mCurrentAdapter = mDeviceListAdapter;

        // Get a support ActionBar corresponding to this toolbar
        ActionBar ab = getSupportActionBar();

        // Enable the Up button
        ab.setDisplayHomeAsUpEnabled(true);

    }

    public void itemSelected(CustomListItem data)
    {
        Log.d(TAG, "itemSelected " + data.getTitle());
        if (data.getTitle().equals(getString(de.kaisersite.mylibrary.R.string.download)))
            {
            for (CustomListItem item: mItemList)
                {
                if (item.getIcon()==R.drawable.ic_load)
                    mFragment.navigateTo(item, bBrowseAVTransport);
                }
            }
        else
            {
            mFragment.navigateTo(data, bBrowseAVTransport);
            }
    }

    @Override
    public void onBackPressed()
    {
        if (mFragment==null || mFragment.goBack())
            super.onBackPressed();
    }

    @Override
    public void onDisplayDevices()
    {
        runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                getSupportActionBar().setSubtitle("");
                mRecyclerView.setAdapter(mDeviceListAdapter);
            }
        });
        mCurrentAdapter = mDeviceListAdapter;
        if (mDeviceList.size() == 0)
            {
            showWaitIndicator(false);
            }
        else
            {
            hideWaitIndicator();
            }
    }

    @Override
    public void onDisplayDirectories(final String title)
    {
        runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                getSupportActionBar().setSubtitle(title);
                hideWaitIndicator();
                mItemList.clear();
                mItemList.add(new CustomListItem(R.drawable.ic_folderup, "..", ""));
                CustomListItem cli = new CustomListItem(R.drawable.ic_folder, "Loading", "");
                cli.setLoading(true);
                mItemList.add(cli);
                mRecyclerView.setAdapter(mItemListAdapter);
                mCurrentAdapter = mItemListAdapter;
                mItemListAdapter.notifyDataSetChanged();
            }
        });
    }

    @Override
    public void onDisplayItems(final ArrayList<ItemModel> items)
    {
        runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                mItemList.clear();
                mItemList.add(new CustomListItem(R.drawable.ic_folderup, "..", ""));
                for (ItemModel a:items)
                    {
                    if (!a.isContainer())
                        {
                        mItemList.add(new CustomListItem(R.drawable.ic_loadall, getString(R.string.download), getString(R.string.all_in_this_folder)));
                        break;
                        }
                    }
                mItemList.addAll(items);
                mItemListAdapter.notifyDataSetChanged();
            }
        });
    }

    @Override
    public void onDisplayItemsError(final String error)
    {
        runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                mItemList.clear();
                mItemList.add(new CustomListItem(R.drawable.ic_folderup, "..", ""));
                mItemList.add(new CustomListItem(
                        R.drawable.ic_warning,
                        getResources().getString(R.string.upnp_folder_info_errorlist_folders),
                        error));
                mItemListAdapter.notifyDataSetChanged();
            }
        });
    }

    @Override
    public void onContentDeviceAdded(final DeviceModel device)
    {
        if (bBrowseAVTransport)
            return;
        runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                mWaitForDevices.setVisibility(View.INVISIBLE);
                mWaitForWLAN.setVisibility(View.INVISIBLE);
                int position = mDeviceList.indexOf(device);
                if (position >= 0)
                    {
                    mDeviceList.remove(device);
                    mDeviceList.add(position, device);
                    mDeviceListAdapter.notifyItemChanged(position);

                    }
                else
                    {
                    mDeviceList.add(device);
                    mDeviceListAdapter.notifyItemInserted(mDeviceList.size()-1);
                    }
            }
        });
    }

    @Override
    public void onAVTransportDeviceAdded(final DeviceModel device)
    {
        if (!bBrowseAVTransport)
            return;
        runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                mWaitForDevices.setVisibility(View.INVISIBLE);
                mWaitForWLAN.setVisibility(View.INVISIBLE);
                int position = mDeviceList.indexOf(device);
                if (position >= 0)
                    {
                    mDeviceList.remove(device);
                    mDeviceList.add(position, device);
                    mDeviceListAdapter.notifyItemChanged(position);

                    }
                else
                    {
                    mDeviceList.add(device);
                    mDeviceListAdapter.notifyItemInserted(mDeviceList.size()-1);
                    }
            }
        });
    }

    @Override
    public void onDeviceRemoved(final DeviceModel device)
    {
        runOnUiThread(new Runnable()
        {
            @Override
            public void run()
            {
                int position = mDeviceList.indexOf(device);
                mDeviceList.remove(device);
                mDeviceListAdapter.notifyItemRemoved(position);
                if (mDeviceList.size() == 0 && mCurrentAdapter == mDeviceListAdapter)
                    {
                    showWaitIndicator(false);
                    }
            }
        });
    }

    // Receive Network Status
    private BroadcastReceiver mMessageReceiver = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            if ("available".equals(intent.getStringExtra("status")))
                {
                startUPNP();
                }
            else if ("lost".equals(intent.getStringExtra("status")))
                {
                pauseUPNP();
                }
            else if ("timeout_lost".equals(intent.getStringExtra("status")))
                {
                finish();
                }
            else if ("timeout".equals(intent.getStringExtra("status")))
                {
                PackageManager packageManager = getPackageManager();
                Intent intent2 = new Intent(Settings.ACTION_WIFI_SETTINGS);
                if (intent2.resolveActivity(packageManager) != null)
                    {
                    startActivity(intent2);
                    }
                else
                    {
                    finish();
                    }
                }
        }
    };

    private void pauseUPNP()
    {
        if (mDeviceList.size() == 0 && mCurrentAdapter == mDeviceListAdapter)
            {
            showWaitIndicator(true);
            }
        else
            {
            hideWaitIndicator();
            }
        mItemList.clear();
        mItemList.add(new CustomListItem(R.drawable.ic_folderup, "..", ""));
        CustomListItem cli = new CustomListItem(R.drawable.ic_folder, getString(R.string.upnp_folder_wait_for_wifi), "");
        cli.setLoading(true);
        mItemList.add(cli);
        mItemListAdapter.notifyDataSetChanged();
    }

    private void startUPNP()
    {
        if (mDeviceList.size() == 0 && mCurrentAdapter == mDeviceListAdapter)
            {
            showWaitIndicator(false);
            }
        else
            {
            hideWaitIndicator();
            }
        if (mFragment != null)
            {
            mFragment.refreshCurrent(true);
            }
        else
            {
            FragmentManager fragmentManager = getFragmentManager();
            mFragment = new ContentDirectoryBrowseTaskFragment();
            fragmentManager.beginTransaction().add(mFragment, "task").commit();
            }
    }


    @Override
    public void onResume()
    {
        super.onResume();
        LocalBroadcastManager.getInstance(this).registerReceiver(mMessageReceiver, new IntentFilter("wlan_status"));
        if (NetworkManaging.getInstance(this).isInternetAvailable())
            {
            startUPNP();
            }
        else
            {
            pauseUPNP();
            }
        NetworkManaging.getInstance(this).addRegistration();
    }

    @Override
    protected void onPause()
    {
        super.onPause();
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mMessageReceiver);
        NetworkManaging.getInstance(this).removeRegistration();
    }

    @Override
    public void onRefresh()
    {
        mFragment.refreshCurrent(false);
        mSwipeToRefreshLayout.setRefreshing(false);
    }

    private void showWaitIndicator(boolean bWifi)
    {
        if (bWifi)
            {
            mWaitForDevices.setVisibility(View.INVISIBLE);
            mWaitForWLAN.setVisibility(View.VISIBLE);
            }
        else
            {
            mWaitForDevices.setVisibility(View.VISIBLE);
            mWaitForWLAN.setVisibility(View.INVISIBLE);
            }

    }

    private void hideWaitIndicator()
    {
        mWaitForDevices.setVisibility(View.INVISIBLE);
        mWaitForWLAN.setVisibility(View.INVISIBLE);
    }
}
