/**
 * Copyright (C) 2018 Achim Kaiser
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.kaisersite.akwearplayer;

import android.app.job.JobScheduler;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Bundle;
import android.provider.Settings;
import androidx.core.content.ContextCompat;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.SimpleItemAnimator;
import androidx.wear.widget.WearableRecyclerView;
import androidx.wear.widget.drawer.WearableActionDrawerView;
import android.support.wearable.activity.WearableActivity;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.Toast;

import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.Tasks;
import com.google.android.gms.wearable.Node;
import com.google.android.gms.wearable.Wearable;
import com.google.android.wearable.intent.RemoteIntent;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.concurrent.ExecutionException;

import de.kaisersite.mylibrary.CurrentDevice;
import de.kaisersite.mylibrary.CustomListItem;
import de.kaisersite.mylibrary.DownloadJobService;
import de.kaisersite.mylibrary.MyMediaService;
import de.kaisersite.mylibrary.MyPlayListAdapter;
import de.kaisersite.mylibrary.PlayListModel;
import de.kaisersite.mylibrary.PlaylistDB;

public class MainActivity extends WearableActivity implements MenuItem.OnMenuItemClickListener, MyPlayListAdapter.Controller
{
    private static final String TAG = "MainActivity";
    private static final int CONFIRMATION_DELETE = 444;
    private static final int CONFIRMATION_STARTPHONE = 555;
    private WearableActionDrawerView mWearableActionDrawer;
    private LinearLayout mAddbutton;

    PlaylistDB playlist;

    private MyPlayListAdapter mItemListAdapter;
    private WearableRecyclerView mRecyclerView;
    private String playlistid;

    @Override
    protected void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);
        if (getIntent().getBooleanExtra("cancel", false))
            {
            JobScheduler js = (JobScheduler) getSystemService(Context.JOB_SCHEDULER_SERVICE);
            js.cancelAll();
            }

        setContentView(R.layout.activity_main);
        mWearableActionDrawer = findViewById(R.id.bottom_action_drawer);
        mWearableActionDrawer.setOnMenuItemClickListener(this);
        mWearableActionDrawer.setPeekOnScrollDownEnabled(true);
        Menu menu = mWearableActionDrawer.getMenu();
        menu.clear();
        Intent intent=new Intent();
        intent.setClassName("com.google.android.wearable.app","com.google.android.clockwork.home.media.MediaControlActivity");
        if (getPackageManager().queryIntentActivities(intent,PackageManager.MATCH_ALL).size()==0)
            getMenuInflater().inflate(R.menu.action_drawer_menu, menu);
        else
            getMenuInflater().inflate(R.menu.action_drawer_menu_controls, menu);
        mWearableActionDrawer.getController().peekDrawer();

        mAddbutton = findViewById(R.id.addbutton);
        mAddbutton.setOnClickListener(new View.OnClickListener()
        {
            @Override
            public void onClick(View view)
            {
                Intent intent = new Intent(MainActivity.this, UPNPActivity.class);
                startActivity(intent);
            }
        });
        mRecyclerView = findViewById(R.id.recycler_launcher_view);

        mRecyclerView.setLayoutManager(
                new LinearLayoutManager(this));
        mRecyclerView.setEdgeItemsCenteringEnabled(true);
        ((SimpleItemAnimator) mRecyclerView.getItemAnimator()).setSupportsChangeAnimations(false);
        createAdapter();
        mRecyclerView.requestFocus();
    }

    private void createAdapter()
    {
        playlist = new PlaylistDB(this, PlaylistDB.getCurrentPlaylistId(this));
        mItemListAdapter = new MyPlayListAdapter(this, this,null,playlist,false);
        mItemListAdapter.setHasStableIds(true);
        mRecyclerView.setAdapter(mItemListAdapter);
    }

    @Override
    public boolean onMenuItemClick(MenuItem item)
    {
        mWearableActionDrawer.getController().closeDrawer();
        switch (item.getItemId())
            {
            case R.id.action_add:
            {
            Intent intent = new Intent(this, UPNPActivity.class);
            startActivity(intent);
            return true;
            }
            case R.id.action_volume:
            {
            Intent intent = new Intent(Settings.ACTION_SOUND_SETTINGS);
            startActivity(intent);
            return true;
            }
            case R.id.action_controls:
            {
            Intent intent = new Intent();
            intent.setClassName("com.google.android.wearable.app","com.google.android.clockwork.home.media.MediaControlActivity");
            intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK );
            startActivity(intent);
            return true;
            }
            case R.id.action_move:
            {
            Menu menu = mWearableActionDrawer.getMenu();
            menu.clear();
            getMenuInflater().inflate(R.menu.action_drawer_edit, menu);
            mWearableActionDrawer.getController().peekDrawer();
            mWearableActionDrawer.setIsLocked(true);
            mItemListAdapter.setEditmode(MyPlayListAdapter.EDIT_MOVE);
            return true;
            }
            case R.id.action_delete:
            {
            closePlayer();
            stopDownload();
            Menu menu = mWearableActionDrawer.getMenu();
            menu.clear();
            getMenuInflater().inflate(R.menu.action_drawer_edit, menu);
            mWearableActionDrawer.getController().peekDrawer();
            mWearableActionDrawer.setIsLocked(true);
            mItemListAdapter.setEditmode(MyPlayListAdapter.EDIT_DELETE);
            return true;
            }
            case R.id.action_delete_playlist:
            {
            closePlayer();
            stopDownload();
            Intent intent = new Intent(this,Confirmation.class);
            intent.putExtra("text",getResources().getString(R.string.action_delete_playlist));
            startActivityForResult(intent,CONFIRMATION_DELETE);
            return true;
            }
//            case R.id.action_debug_process:
//            {
//            Tools.debugProcess();
//            return true;
//            }
            case R.id.action_cast:
            {
            if (CurrentDevice.getId(MainActivity.this)==null)
                {
                Intent intent = new Intent(this, UPNPActivity.class);
                intent.putExtra("BrowseAVTransport", true);
                startActivity(intent);
                return true;
                }
            else
                {
                CurrentDevice.setDevice( this,null,null);
                Intent intent = new Intent(MyMediaService.CLOSE_PLAYER_STOP_CLOSE);
                sendBroadcast(intent);
                Toast.makeText(this,R.string.stop_casting,Toast.LENGTH_SHORT).show();
                }
            }
            case R.id.action_send:
            {
            sendItems();
            return true;
            }
            case R.id.action_start_on_phone:
            {
            stopPhonePlaying();
            closePlayer();
            Intent intent = new Intent(this,Confirmation.class);
            intent.putExtra("text",getResources().getString(R.string.action_start_on_phone));
            startActivityForResult(intent,CONFIRMATION_STARTPHONE);
            return true;
            }
            case R.id.action_leave:
            {
            if (playlist.size() == 0)
                mAddbutton.setVisibility(View.VISIBLE);
            Menu menu = mWearableActionDrawer.getMenu();
            menu.clear();
            Intent intent=new Intent();
            intent.setClassName("com.google.android.wearable.app","com.google.android.clockwork.home.media.MediaControlActivity");
            if (getPackageManager().queryIntentActivities(intent,PackageManager.MATCH_ALL).size()==0)
                getMenuInflater().inflate(R.menu.action_drawer_menu, menu);
            else
                getMenuInflater().inflate(R.menu.action_drawer_menu_controls, menu);
            mWearableActionDrawer.getController().peekDrawer();
            mWearableActionDrawer.setIsLocked(false);
            mItemListAdapter.setEditmode(MyPlayListAdapter.EDIT_PLAY);
            return true;
            }
            case R.id.action_legal:
            {
            Intent intent = new Intent(Intent.ACTION_VIEW)
                    .addCategory(Intent.CATEGORY_BROWSABLE)
                    .setData(Uri.parse("https://www.kaisersite.de/akwearplayer"));
            Toast.makeText(this,R.string.toast_legal_link,Toast.LENGTH_SHORT).show();
            RemoteIntent.startRemoteActivity(this, intent, null);
            return true;
            }
            case R.id.action_version:
            {
            Intent intent = new Intent(this, VersionActivity.class);
            startActivity(intent);
            return true;
            }
            }
        return false;
    }

    private void closePlayer()
    {
        Intent intent = new Intent(MyMediaService.CLOSE_PLAYER);
        sendBroadcast(intent);
    }

    private void stopDownload()
    {
        JobScheduler js = (JobScheduler) getSystemService(Context.JOB_SCHEDULER_SERVICE);
        js.cancelAll();
        PlaylistDB.setAllLoadingToError(this);
        mItemListAdapter.notifyDataSetChanged();
    }


    public void itemSelected(CustomListItem data)
    {
        PlayListModel plItem = (PlayListModel) data;
        if (mItemListAdapter.getEditmode() == MyPlayListAdapter.EDIT_DELETE)
            {
            new File(plItem.getFile()).delete();
            int pos=playlist.getPositon(plItem.getId());
            playlist.deleteItem(plItem.getId());
            mItemListAdapter.notifyItemRemoved(pos);
            Log.d(TAG, "itemDeleted " + data.getTitle());
            }
        else if (mItemListAdapter.getEditmode() == MyPlayListAdapter.EDIT_MOVE)
            {
            int pos=playlist.getPositon(plItem.getId());
            playlist.moveToLast(plItem.getId());
            mItemListAdapter.notifyItemMoved(pos,playlist.size()-1);
            updatePlayNext();
            Log.d(TAG, "itemMoved" + data.getTitle());
            }
        else
            {
            if (plItem.getState() == PlayListModel.STATE_LOADING)
                {
                Toast.makeText(this, "Still loading", Toast.LENGTH_SHORT).show();
                return;
                }
            else if (plItem.getState() == PlayListModel.STATE_ERROR)
                {
                Intent intent = new Intent()
                        .putExtra("url", plItem.getUrl())
                        .putExtra("title", plItem.getTitle())
                        .putExtra("desc", plItem.getDescription())
                        .putExtra("file", plItem.getFile())
                        .putExtra("id", plItem.getId())
                        .putExtra("playlistid", PlaylistDB.getCurrentPlaylistId(this));
                playlist.setState(plItem.getId(), PlayListModel.STATE_LOADING);
                DownloadJobService.enqueueWork(this, intent);
                mItemListAdapter.notifyItemChanged(playlist.getPositon(plItem.getId()));
                Toast.makeText(this, "Loading...", Toast.LENGTH_SHORT).show();
                return;
                }
            else
                {
                startPlaying(plItem.getId());
                }
            }
    }

    public void startPlaying(String id)
    {
        mItemListAdapter.setPlayingId(id, MyPlayListAdapter.CAST_PREPARE);
        Intent intent = new Intent(this, MyMediaService.class);
        intent.putExtra("id", id);
        intent.putExtra("playlistid", PlaylistDB.getCurrentPlaylistId(this));
        ContextCompat.startForegroundService(this, intent);
    }


    //broadcast intent for update list
    private BroadcastReceiver mMessageReceiverUpdateDownload = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {

            mItemListAdapter.notifyDataSetChanged();
        }
    };

    //broadcast intent for playing from MyMediaService
    private BroadcastReceiver mMessageReceiverUpdatePlaying = new BroadcastReceiver()
    {
        @Override
        public void onReceive(Context context, Intent intent)
        {
            playlistid =intent.getStringExtra("id")==null ? null : intent.getStringExtra("playlistid");

            if (PlaylistDB.getCurrentPlaylistId(context).equals(intent.getStringExtra("playlistid")))
                {
                mItemListAdapter.setPlayingId(intent.getStringExtra("id"), intent.getBooleanExtra("paused", true)?MyPlayListAdapter.CAST_PAUSE:MyPlayListAdapter.CAST_PLAY);
                }
            else
                {
                mItemListAdapter.setPlayingId(null, MyPlayListAdapter.CAST_NONE);
                }
        }
    };


    @Override
    public void onResume()
    {
        super.onResume();
        LocalBroadcastManager.getInstance(this).registerReceiver(mMessageReceiverUpdateDownload, new IntentFilter(DownloadJobService.UPDATE_DOWNLOAD));
        LocalBroadcastManager.getInstance(this).registerReceiver(mMessageReceiverUpdatePlaying, new IntentFilter(MyMediaService.UPDATE_PLAYING));

        mItemListAdapter.setPlayingId(null, MyPlayListAdapter.CAST_NONE);
        mItemListAdapter.notifyDataSetChanged();

        Intent bcintent = new Intent(MyMediaService.REQUEST_PLAYSTATUS);
        //send broadcast
        LocalBroadcastManager.getInstance(this).sendBroadcast(bcintent);

        mWearableActionDrawer.getController().peekDrawer();
        mAddbutton.setVisibility(playlist.size() == 0 ? View.VISIBLE : View.INVISIBLE);
    }

    @Override
    protected void onPause()
    {
        super.onPause();
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mMessageReceiverUpdateDownload);
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mMessageReceiverUpdatePlaying);
    }

    @Override
    protected void onNewIntent(Intent intent)
    {
        super.onNewIntent(intent);
        if (intent.getBooleanExtra("cancel", false))
            {
            JobScheduler js = (JobScheduler) getSystemService(Context.JOB_SCHEDULER_SERVICE);
            js.cancelAll();
            }
    }

    private void sendItems()
    {
        // Trigger an AsyncTask that will query for a list of connected nodes and send a
        // "start-activity" message to each connected node.
        new StartWearableActivityTask().execute("/download");

    }

    private void startPhonePlaying()
    {
        // Trigger an AsyncTask that will query for a list of connected nodes and send a
        // "start-activity" message to each connected node.
        new StartWearableActivityTask().execute("/startplaying");

    }

    private void stopPhonePlaying()
    {
        // Trigger an AsyncTask that will query for a list of connected nodes and send a
        // "start-activity" message to each connected node.
        new StartWearableActivityTask().execute("/stopplaying");

    }

    private class StartWearableActivityTask extends AsyncTask<String, Void, Void>
    {
        @Override
        protected Void doInBackground(String... types)
        {
            String type=types[0];
            //first get all the nodes, ie connected wearable devices.
            Task<List<Node>> nodeListTask =
                    Wearable.getNodeClient(getApplicationContext()).getConnectedNodes();
            try
                {
                // Block on a task and get the result synchronously (because this is on a background
                // thread).
                List<Node> nodes = Tasks.await(nodeListTask);
                //Now send the message to each device.
                for (Node node : nodes)
                    {
                    if (type.equals("/download"))
                        {
                        for (int i = 0; i < playlist.size(); i++)
                            {
                            byte[] data;
                            try
                                {
                                JSONObject json = new JSONObject();
                                String id = playlist.get(i).getId();
                                json.put("title",playlist.getItem(id).getTitle());
                                json.put("description",playlist.getItem(id).getDescription());
                                json.put("url",playlist.getItem(id).getUrl());
                                json.put("description2",playlist.getItem(id).getDescription2());
                                json.put("duration",playlist.getItem(id).getDuration());
                                json.put("url",playlist.getItem(id).getUrl());
                                data = json.toString().getBytes("UTF-8");
                                }
                            catch (JSONException | UnsupportedEncodingException e)
                                {
                                continue;
                                }
                            Task<Integer> sendMessageTask =
                                    Wearable.getMessageClient(MainActivity.this).sendMessage(node.getId(), "/download", data);

                            try
                                {
                                // Block on a task and get the result synchronously (because this is on a background
                                // thread).
                                Integer result = Tasks.await(sendMessageTask);
                                Log.v(TAG, "SendThread: message send to " + node.getDisplayName());

                                }
                            catch (ExecutionException exception)
                                {
                                Log.e(TAG, "Task failed: " + exception);

                                }
                            catch (InterruptedException exception)
                                {
                                Log.e(TAG, "Interrupt occurred: " + exception);
                                }

                            }

                        }
                    else if (type.equals("/startplaying"))
                        {
                        Task<Integer> sendMessageTask =
                                Wearable.getMessageClient(MainActivity.this).sendMessage(node.getId(), "/startplaying", new byte[]{});

                        try
                            {
                            // Block on a task and get the result synchronously (because this is on a background
                            // thread).
                            Integer result = Tasks.await(sendMessageTask);
                            Log.v(TAG, "SendThread: message send to " + node.getDisplayName());

                            }
                        catch (ExecutionException exception)
                            {
                            Log.e(TAG, "Task failed: " + exception);

                            }
                        catch (InterruptedException exception)
                            {
                            Log.e(TAG, "Interrupt occurred: " + exception);
                            }

                        }
                    else if (type.equals("/stopplaying"))
                        {
                        Task<Integer> sendMessageTask =
                                Wearable.getMessageClient(MainActivity.this).sendMessage(node.getId(), "/stopplaying", new byte[]{});

                        try
                            {
                            // Block on a task and get the result synchronously (because this is on a background
                            // thread).
                            Integer result = Tasks.await(sendMessageTask);
                            Log.v(TAG, "SendThread: message send to " + node.getDisplayName());

                            }
                        catch (ExecutionException exception)
                            {
                            Log.e(TAG, "Task failed: " + exception);

                            }
                        catch (InterruptedException exception)
                            {
                            Log.e(TAG, "Interrupt occurred: " + exception);
                            }

                        }

                    }
                }
            catch (ExecutionException exception)
                {
                Log.e(TAG, "Task failed: " + exception);

                }
            catch (InterruptedException exception)
                {
                Log.e(TAG, "Interrupt occurred: " + exception);
                }
            return null;
        }
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data)
    {
        super.onActivityResult(requestCode, resultCode, data);
        if (requestCode==CONFIRMATION_DELETE)
            {
            if (resultCode==RESULT_OK)
                {
                int itemsize=playlist.size();
                for (int i = 0; i<itemsize; i++)
                    {
                    PlayListModel plm = playlist.get(0);
                    new File(plm.getFile()).delete();
                    playlist.deleteItem(plm.getId());
                    }
                String newpid=PlaylistDB.deletePlaylist(MainActivity.this,PlaylistDB.getCurrentPlaylistId(MainActivity.this));
                PlaylistDB.setCurrentPlaylistId(MainActivity.this,newpid);
                createAdapter();
                }
            }
        else if (requestCode==CONFIRMATION_STARTPHONE)
            {
            if (resultCode==RESULT_OK)
                startPhonePlaying();
            }
    }

    private void updatePlayNext()
    {
        Intent bcintent = new Intent(MyMediaService.UPDATE_NEXT);
        LocalBroadcastManager.getInstance(this).sendBroadcast(bcintent);
    }



}
