/**
 * Copyright (C) 2018 Achim Kaiser
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.kaisersite.mylibrary;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;

import java.util.ArrayList;


public class PlaylistDB
{
    private final String playlistid;
    private Context context;

    public PlaylistDB(Context context, String pid)
    {
        this.context = context;
        this.playlistid = pid;
    }

    /**
     * Adds an item and set loading to true
     *
     * @param url
     * @param file
     * @param title
     * @param description
     * @return
     */
    public boolean addItem(String url, String id, String file, String title, String description, String description2, long duration)
    {
        if (contains(id))
            return false;
        SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
        String liststr = sp.getString("keylist", null);
        if (liststr == null)
            {
            liststr = "" + id;
            }
        else
            {
            liststr += "," + id;
            }

        SharedPreferences.Editor ed = sp.edit();
        ed.putString("keylist", liststr);
        ed.putString("url." + id, url);
        ed.putString("file." + id, file);
        ed.putString("title." + id, title);
        ed.putString("description." + id, description);
        ed.putString("description2." + id, description2);
        ed.putLong("duration." + id, duration);
        ed.putInt("state." + id, PlayListModel.STATE_LOADING);
        ed.commit();
        PlayListModel pm = new PlayListModel(
                R.drawable.ic_file,
                PlayListModel.STATE_LOADING,
                id,
                url,
                file,
                title,
                description, sp.getString("description2." + id, null), sp.getLong("duration." + id, -1));
        return true;
    }

    public PlayListModel get(int index)
    {
        SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
        String liststr = sp.getString("keylist", null);
        if (liststr == null)
            {
            return null;
            }
        String[] arrayids;
        if (!liststr.contains(","))
            {
            arrayids = new String[1];
            arrayids[0] = liststr;
            }
        else
            {
            arrayids = liststr.split(",");
            }
        return getItem(arrayids[index]);
    }

    public int size()
    {
        SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
        String liststr = sp.getString("keylist", null);
        if (liststr == null)
            {
            return 0;
            }
        String[] arrayids;
        if (!liststr.contains(","))
            {
            return 1;
            }
        else
            {
            return liststr.split(",").length;
            }
    }

    public PlayListModel getItem(String id)
    {
        if (!contains(id))
            return null;
        SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
        return new PlayListModel(
                R.drawable.ic_file,
                sp.getInt("state." + id, 0),
                id,
                sp.getString("url." + id, null),
                sp.getString("file." + id, null),
                sp.getString("title." + id, null),
                sp.getString("description." + id, null),
                sp.getString("description2." + id, null),
                sp.getLong("duration." + id, -1)
        );
    }

    public String getNextId(String id)
    {
        if (!contains(id))
            return null;
        SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
        String liststr = sp.getString("keylist", null);
        String[] arrayids;
        if (liststr.contains(","))
            {
            arrayids = liststr.split(",");
            }
        else
            {
            arrayids = new String[1];
            arrayids[0] = liststr;
            }
        for (int i = 0; i < arrayids.length - 1; i++)
            {
            if (id.equals(arrayids[i]))
                {
                int j = i + 1;
                while (j < arrayids.length && getItem(arrayids[j]).getState() != PlayListModel.STATE_LOADED)
                    j++;
                if (j < arrayids.length)
                    return arrayids[j];
                return null;
                }
            }
        return null;
    }

    public String getPrevId(String id)
    {
        if (!contains(id))
            return null;
        SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
        String liststr = sp.getString("keylist", null);
        String[] arrayids;
        if (liststr.contains(","))
            {
            arrayids = liststr.split(",");
            }
        else
            {
            arrayids = new String[1];
            arrayids[0] = liststr;
            }
        for (int i = 1; i < arrayids.length; i++)
            {
            if (id.equals(arrayids[i]))
                {
                int j = i - 1;
                while (j >= 0 && getItem(arrayids[j]).getState() != PlayListModel.STATE_LOADED)
                    j--;
                if (j >= 0)
                    return arrayids[j];
                return null;
                }
            }
        return null;
    }

    public int getPositon(String id)
    {
        SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
        String liststr = sp.getString("keylist", null);
        if (liststr == null)
            return -1;
        String[] arrayids;
        if (liststr.contains(","))
            {
            arrayids = liststr.split(",");
            }
        else
            {
            arrayids = new String[1];
            arrayids[0] = liststr;
            }
        for (int i = 0; i < arrayids.length; i++)
            {
            if (id.equals(arrayids[i]))
                return i;
            }
        return -1;
    }

    public void deleteItem(String id)
    {
        int pos = getPositon(id);
        if (pos >= 0)
            {
            SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
            String liststr = sp.getString("keylist", null);
            SharedPreferences.Editor ed = sp.edit();
            if (liststr.equals(id))
                ed.remove("keylist");
            else
                {
                String searchpattern = "," + liststr + ",";
                String replaced = searchpattern.replace("," + id + ",", ",");
                liststr = replaced.substring(1, replaced.length() - 1);
                ed.putString("keylist", liststr);
                }
            ed.remove("url." + id);
            ed.remove("file." + id);
            ed.remove("title." + id);
            ed.remove("description." + id);
            ed.remove("description2." + id);
            ed.remove("duration." + id);
            ed.remove("status." + id);
            ed.commit();
            }
    }

    public void setState(String id, int state)
    {
        int pos = getPositon(id);
        if (pos >= 0)
            {
            SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
            SharedPreferences.Editor ed = sp.edit();
            ed.putInt("state." + id, state);
            ed.commit();
            }
    }

    public boolean contains(String id)
    {
        int pos = getPositon(id);
        return pos >= 0;
    }

    public void setAllLoadingToError()
    {
        SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
        String liststr = sp.getString("keylist", null);
        if (liststr == null)
            return;
        String[] arrayids;
        if (liststr.contains(","))
            {
            arrayids = liststr.split(",");
            }
        else
            {
            arrayids = new String[1];
            arrayids[0] = liststr;
            }
        SharedPreferences.Editor ed = sp.edit();
        for (int i = 0; i < arrayids.length; i++)
            {
            if (sp.getInt("state." + arrayids[i], PlayListModel.STATE_LOADING) == PlayListModel.STATE_LOADING)
                {
                ed.putInt("state." + arrayids[i], PlayListModel.STATE_ERROR);
                }
            }
        ed.commit();
    }

    public void moveToLast(String id)
    {
        int pos = getPositon(id);
        if (pos >= 0)
            {
            SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
            String liststr = sp.getString("keylist", null);
            if (liststr.equals(id))
                return;
            SharedPreferences.Editor ed = sp.edit();
            String searchpattern = "," + liststr + ",";
            String replaced = searchpattern.replace("," + id + ",", ",");
            liststr = replaced.substring(1, replaced.length() - 1);
            ed.putString("keylist", liststr + "," + id);
            ed.apply();
            }
    }

    public String existsItem(String title, String description)
    {
        if (title == null || description == null)
            return null;
        SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
        String liststr = sp.getString("keylist", null);
        if (liststr == null)
            return null;
        String[] arrayids;
        if (liststr.contains(","))
            {
            arrayids = liststr.split(",");
            }
        else
            {
            arrayids = new String[1];
            arrayids[0] = liststr;
            }
        for (int i = 0; i < arrayids.length; i++)
            {
            if (title.equals(sp.getString("title." + arrayids[i], null)) && description.equals(sp.getString("description." + arrayids[i], null)))
                {
                return arrayids[i];
                }
            }
        return null;
    }

    public void move(int oldPosition, int newPosition)
    {
        SharedPreferences sp = context.getSharedPreferences("filelist" + playlistid, Context.MODE_PRIVATE);
        String liststr = sp.getString("keylist", null);
        if (liststr == null)
            return;
        String[] arrayids;
        if (liststr.contains(","))
            {
            arrayids = liststr.split(",");
            }
        else
            {
            arrayids = new String[1];
            arrayids[0] = liststr;
            }

        ArrayList<String> arraylist = new ArrayList<>();
        for (int i = 0; i < arrayids.length; i++)
            {
            arraylist.add(arrayids[i]);
            }
        String targetUser = arraylist.get(oldPosition);
        arraylist.remove(oldPosition);
        arraylist.add(newPosition, targetUser);
        liststr = "";
        for (int i = 0; i < arraylist.size(); i++)
            {
            liststr += arraylist.get(i) + ",";
            }
        liststr = liststr.substring(0, liststr.length() - 1);
        SharedPreferences.Editor ed = sp.edit();
        ed.putString("keylist", liststr);
        ed.commit();
    }

    public static void setAllLoadingToError(Context context)
    {
        String[] arrayids = getArray(context);
        for (int i = 0; i < arrayids.length; i++)
            {
            new PlaylistDB(context, arrayids[i]).setAllLoadingToError();
            }
    }


    public static String getCurrentPlaylistId(Context context)
    {
        SharedPreferences sp = context.getSharedPreferences("playlists", Context.MODE_PRIVATE);
        return sp.getString("currentlist", "1");
    }

    public static void setCurrentPlaylistId(Context context, String pid)
    {
        SharedPreferences sp = context.getSharedPreferences("playlists", Context.MODE_PRIVATE);
        SharedPreferences.Editor ed = sp.edit();
        ed.putString("currentlist", pid);
        ed.commit();
    }


    public static String[] getArray(Context context)
    {
        SharedPreferences sp = context.getSharedPreferences("playlists", Context.MODE_PRIVATE);
        String liststr = sp.getString("playlistids", "1");
        String[] arrayids;
        if (liststr.contains(","))
            {
            arrayids = liststr.split(",");
            }
        else
            {
            arrayids = new String[1];
            arrayids[0] = liststr;
            }

        return arrayids;
    }

    public static String getName(Context context, String pid)
    {
        SharedPreferences sp = context.getSharedPreferences("playlists", Context.MODE_PRIVATE);
        return sp.getString("name." + pid, "Playlist 1");
    }

    public static boolean contains(Context context, String pid)
    {
        SharedPreferences sp = context.getSharedPreferences("playlists", Context.MODE_PRIVATE);
        String liststr = sp.getString("playlistids", "1");
        return ("," + liststr + ",").contains("," + pid + ",");
    }

    public static String addPlaylist(Context context)
    {
        int pid = 1;
        while (PlaylistDB.contains(context, "" + pid))
            pid++;
        SharedPreferences sp = context.getSharedPreferences("playlists", Context.MODE_PRIVATE);
        String liststr = sp.getString("playlistids", "1");
        liststr += "," + pid;
        SharedPreferences.Editor ed = sp.edit();
        ed.putString("playlistids", liststr);
        ed.putString("name." + pid, "Playlist " + pid);
        ed.apply();
        return ""+pid;
    }

    public static void setPlaylistName(Context context, String pid, String text)
    {
        SharedPreferences sp = context.getSharedPreferences("playlists", Context.MODE_PRIVATE);
        if (PlaylistDB.contains(context,pid))
            {
            SharedPreferences.Editor ed = sp.edit();
            ed.putString("name." + pid, text);
            ed.apply();
            }
    }

    public static String deletePlaylist(Context context, String pid)
    {
        String ret = "1";
        String[] arrayids = getArray(context);
        SharedPreferences sp = context.getSharedPreferences("playlists", Context.MODE_PRIVATE);
        SharedPreferences.Editor ed = sp.edit();
        if (arrayids.length == 1)
            {
            ed.remove("playlistids");
            }
        else
            {
            String liststr = ",";
            boolean found = false;
            for (int i = 0; i < arrayids.length; i++)
                {
                if (!pid.equals(arrayids[i]))
                    {
                    liststr += arrayids[i] + ",";
                    if (!found)
                        ret = arrayids[i];
                    }
                else
                    {
                    if (i + 1 < arrayids.length)
                        ret = arrayids[i + 1];
                    found=true;
                    }
                }
            liststr = liststr.substring(1, liststr.length() - 1);
            ed.putString("playlistids", liststr);
            }
        ed.remove("name." + pid);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N)
            {
            context.deleteSharedPreferences("filelist" + pid);
            }
        ed.apply();
        return ret;
    }
}
