/**
 * Copyright (C) 2018 Achim Kaiser
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.kaisersite.mylibrary;

import android.content.Context;
import android.content.Intent;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkRequest;
import android.os.Handler;
import android.os.Message;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import android.util.Log;
import android.widget.Toast;

public class NetworkManaging implements Handler.Callback
{
    private static final String TAG = "NetworkManaging";
    private static NetworkManaging networkmanaging;
    private final Handler mHandler;
    private int usage = 0;

    private Context appcontext;

    private final int MESSAGE_LOST_TIMEOUT = 1;
    private final int MESSAGE_CONNECTIVITY_TIMEOUT = 2;
    private final long NETWORK_CONNECTIVITY_TIMEOUT_MS = 10000;
    private boolean internetavailable = false;

    public synchronized static NetworkManaging getInstance(Context context)
    {
        if (networkmanaging == null)
            networkmanaging = new NetworkManaging(context.getApplicationContext());
        return networkmanaging;
    }

    private NetworkManaging(Context context)
    {
        appcontext = context;
        mHandler = new Handler(this);
    }

    public synchronized void addRegistration()
    {
        if (usage == 0)
            {
            internetavailable = false;
            NetworkRequest request;
            if (BuildConfig.DEBUG)
                {
                request = new NetworkRequest.Builder()
                    .addTransportType(NetworkCapabilities.TRANSPORT_WIFI)
                    .addTransportType(NetworkCapabilities.TRANSPORT_CELLULAR) //for emulator
                    //                .addCapability(NetworkCapabilities.NET_CAPABILITY_NOT_METERED)
                    //                .addCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)
                    .build();
                }
            else
                {
                request = new NetworkRequest.Builder().addTransportType(NetworkCapabilities.TRANSPORT_WIFI).build();
                }
            ConnectivityManager mConnectivityManager = (ConnectivityManager) appcontext.getSystemService(Context.CONNECTIVITY_SERVICE);
            mConnectivityManager.requestNetwork(request, mNetworkCallback);
            Log.d(TAG, "MESSAGE_CONNECTIVITY_TIMEOUT start");
            mHandler.sendMessageDelayed(
                    mHandler.obtainMessage(MESSAGE_CONNECTIVITY_TIMEOUT),
                    NETWORK_CONNECTIVITY_TIMEOUT_MS);
            }
         usage++;
        Log.d(TAG, "addRegistration usage " + (usage - 1) + "->" + usage);
        //Toast.makeText(appcontext, "usage- "+usage, Toast.LENGTH_SHORT).show();

    }

    public synchronized void removeRegistration()
    {
        if (usage == 1)
            {
            mHandler.removeMessages(MESSAGE_CONNECTIVITY_TIMEOUT);
            mHandler.removeMessages(MESSAGE_LOST_TIMEOUT);
            ConnectivityManager mConnectivityManager = (ConnectivityManager) appcontext.getSystemService(Context.CONNECTIVITY_SERVICE);
            mConnectivityManager.unregisterNetworkCallback(mNetworkCallback);
            mConnectivityManager.bindProcessToNetwork(null);
            sendNotify(false);
            }
        usage--;
        Log.d(TAG, "removeRegistration usage " + (usage + 1) + "->" + usage);
        //Toast.makeText(appcontext, "usage+ "+usage, Toast.LENGTH_SHORT).show();
    }

    @Override
    public boolean handleMessage(Message msg)
    {
        switch (msg.what)
            {
            case MESSAGE_CONNECTIVITY_TIMEOUT:
                Log.d(TAG, "MESSAGE_CONNECTIVITY_TIMEOUT mHandler");
                Toast.makeText(appcontext, "WLAN not available", Toast.LENGTH_SHORT).show();
                Intent intent = new Intent("wlan_status");
                intent.putExtra("status", "timeout");
                LocalBroadcastManager.getInstance(appcontext).sendBroadcast(intent);
                sendNotify(false);
                break;
            case MESSAGE_LOST_TIMEOUT:
                Log.d(TAG, "MESSAGE_LOST_TIMEOUT mHandler");
                Intent intent2 = new Intent("wlan_status");
                intent2.putExtra("status", "timeout_lost");
                LocalBroadcastManager.getInstance(appcontext).sendBroadcast(intent2);
                sendNotify(false);
                break;
            }
        return true;
    }

    private synchronized void sendNotify(boolean internetavailable)
    {
        this.internetavailable = internetavailable;
        notify();
    }

    private ConnectivityManager.NetworkCallback mNetworkCallback = new ConnectivityManager.NetworkCallback()
    {
        @Override
        public void onAvailable(Network network)
        {
            Log.d(TAG, "onAvailable");
            ConnectivityManager mConnectivityManager = (ConnectivityManager) appcontext.getSystemService(Context.CONNECTIVITY_SERVICE);

            if (mConnectivityManager.bindProcessToNetwork(network))
                {
                Log.i("NetworkManaging", "wlan available");
                mHandler.removeMessages(MESSAGE_CONNECTIVITY_TIMEOUT);
                mHandler.removeMessages(MESSAGE_LOST_TIMEOUT);
                //Toast.makeText(appcontext, "wlan available", Toast.LENGTH_SHORT).show();
                Intent intent = new Intent("wlan_status");
                intent.putExtra("status", "available");
                LocalBroadcastManager.getInstance(appcontext).sendBroadcast(intent);
                sendNotify(true);
                }
            else
                {
                Toast.makeText(appcontext, "no android.permission.INTERNET permission", Toast.LENGTH_SHORT).show();
                }
        }

        @Override
        public void onLost(Network network)
        {
            Log.d(TAG, "onAvailable");
            Toast.makeText(NetworkManaging.this.appcontext, "wlan lost", Toast.LENGTH_SHORT).show();
            Intent intent = new Intent("wlan_status");
            intent.putExtra("status", "lost");
            LocalBroadcastManager.getInstance(appcontext).sendBroadcast(intent);
            internetavailable = false;
            mHandler.sendMessageDelayed(
                    mHandler.obtainMessage(MESSAGE_LOST_TIMEOUT),
                    NETWORK_CONNECTIVITY_TIMEOUT_MS);
        }

    };


    public synchronized boolean waitForInternet()
    {
        Log.i(TAG, "waitForInternet");

        if (usage == 0)
            throw new IllegalArgumentException("no Internet");
        try
            {
            if (!internetavailable)
                wait();
            }
        catch (InterruptedException e)
            {
            e.printStackTrace();
            }
        return internetavailable;
    }

    public boolean isInternetAvailable()
    {
        return internetavailable;
    }
}
