/**
 * Copyright (C) 2018 Achim Kaiser
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.kaisersite.mylibrary;

import android.content.Context;
import android.graphics.Bitmap;
import android.os.AsyncTask;
import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;
import androidx.recyclerview.widget.ItemTouchHelper;

import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.TextView;


public class MyPlayListAdapter extends RecyclerView.Adapter<MyPlayListAdapter.ViewHolder>
{

    private static final String TAG = "MyPlayListAdapter";
    public static final int EDIT_PLAY = 1;
    public static final int EDIT_DELETE = 2;
    public static final int EDIT_MOVE = 3;
    private final ItemTouchHelper touchHelper;
    private final int widthImage;
    private final MyCache<Bitmap> cache;
    private static final int cachesize=30;

    private PlaylistDB mDataSet;

    // Custom Controller used to instruct main activity to update {@link Notification} and/or
    // UI for item selected.
    private Controller mController;
    private int editmode = EDIT_PLAY;
    private String playingid = null;
    public static final int CAST_NONE = 0;
    public static final int CAST_PLAY = 1;
    public static final int CAST_PREPARE = 2;
    public static final int CAST_PAUSE = 3;
    private int eShowing = CAST_NONE ;

    private boolean bShowOrder = false;

    private static int number;

    public static class ViewHolder extends RecyclerView.ViewHolder
    {
        public TextView titleView;
        public TextView descriptionView;
        public TextView description2View;
        public ImageView imageView;
        public ImageView imageView2;
        public ImageView orderView;
        public TextView timeView;
        public ProgressBar progressView;
        public String id;
        public String url;

        public ViewHolder(View view)
        {
            super(view);
            imageView = (ImageView) view.findViewById(R.id.icon);
            imageView2 = (ImageView) view.findViewById(R.id.icon2);
            titleView = (TextView) view.findViewById(R.id.title);
            descriptionView = (TextView) view.findViewById(R.id.description);
            description2View = (TextView) view.findViewById(R.id.description2);
            progressView = (ProgressBar) view.findViewById(R.id.progress);
            orderView = (ImageView) view.findViewById(R.id.order);
            timeView = view.findViewById(R.id.time);
            id="";
            url="";
        }

        @Override
        public String toString()
        {
            return (String) titleView.getText();
        }
    }

    public MyPlayListAdapter(Context context, Controller controller, ItemTouchHelper touchHelper, PlaylistDB playList, boolean bShowOrder)
    {
        mDataSet = playList;
        mController = controller;
        this.touchHelper=touchHelper;
        widthImage = (int)context.getResources().getDimension(R.dimen.albumimagewidth);
        this.bShowOrder = bShowOrder;
        cache=new MyCache<>(cachesize);

    }

    @NonNull
    @Override
    public ViewHolder onCreateViewHolder(@NonNull ViewGroup viewGroup, int viewType)
    {
        View view = LayoutInflater.from(viewGroup.getContext())
                .inflate(R.layout.list, viewGroup, false);

        final ViewHolder viewHolder = new ViewHolder(view);
        viewHolder.itemView.setOnClickListener(new View.OnClickListener()
        {

            @Override
            public void onClick(View view)
            {
                try
                    {
                    mController.itemSelected(mDataSet.get(viewHolder.getAdapterPosition()));
                    }
                catch (ArrayIndexOutOfBoundsException e)
                    {}

            }
        });
        viewHolder.orderView.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                if (event.getActionMasked() == MotionEvent.ACTION_DOWN)
                    {
                    touchHelper.startDrag(viewHolder);
                    }
                return false;
            }
        });
        return viewHolder;

    }

    @Override
    public void onBindViewHolder(@NonNull final ViewHolder viewHolder, final int position)
    {

        PlayListModel entry = mDataSet.get(position);

        viewHolder.titleView.setText(entry.getTitle());

        String description = entry.getDescription();
        if (description == null)
            viewHolder.description2View.setVisibility(View.GONE);
        else
            {
            viewHolder.description2View.setVisibility(View.VISIBLE);
            viewHolder.descriptionView.setText(description);
            }

        String description2 = entry.getDescription2();
        viewHolder.description2View.setVisibility(View.GONE);
        if (entry.getState()==PlayListModel.STATE_LOADING)
            {
            viewHolder.imageView.setVisibility(View.INVISIBLE);
            viewHolder.progressView.setVisibility(View.VISIBLE);
            }
        else if (entry.getState()==PlayListModel.STATE_ERROR)
            {
            viewHolder.imageView.setVisibility(View.VISIBLE);
            viewHolder.progressView.setVisibility(View.INVISIBLE);
            viewHolder.imageView.setImageResource(R.drawable.ic_warning);
            }
        else
            {
            Bitmap b = cache.get(entry.getUrl());
            if (entry.getId().equals(playingid) && eShowing!=CAST_PREPARE )
                {
                viewHolder.imageView.setImageResource(eShowing==CAST_PAUSE ? R.drawable.ic_play : R.drawable.ic_pause);
                }
            else if (b==null)
                {
                viewHolder.imageView.setImageResource(entry.getIcon());
                }
            else
                {
                viewHolder.imageView.setImageBitmap(b);
                }
            if (b==null && !cache.containsKey(entry.getUrl()))
                {
                // URL nicht im cache nachladen
                viewHolder.id=entry.getId();
                viewHolder.url=entry.getUrl();
                AlbumImage a = new AlbumImage();
                a.viewHolder=viewHolder;
                a.playListModel = entry;
                a.myPlayListAdapter = this;
                new PicLoad().execute(a);
                }
            viewHolder.imageView.setVisibility(View.VISIBLE);
            if (entry.getId().equals(playingid) && eShowing==CAST_PREPARE )
                {
                viewHolder.progressView.setVisibility(View.VISIBLE);
                }
            else
                {
                viewHolder.progressView.setVisibility(View.INVISIBLE);
                }
            }
        if (entry.getDuration()>=0)
            {
            viewHolder.timeView.setVisibility(View.VISIBLE);
            viewHolder.timeView.setText(viewHolder.timeView.getContext().getString(R.string.timeHHMM,entry.getDuration()/60000,entry.getDuration()%60000/1000));
            }
        else
            {
            viewHolder.timeView.setVisibility(View.GONE);
            }
        if (editmode == EDIT_DELETE)
            {
            viewHolder.imageView.setImageResource(R.drawable.ic_trash);
            if (entry.getState() == PlayListModel.STATE_ERROR)
                {
                viewHolder.imageView2.setVisibility(View.VISIBLE);
                viewHolder.imageView2.setImageResource(R.drawable.ic_warning);
                }
            else
                {
                viewHolder.imageView2.setVisibility(View.INVISIBLE);
                }
            viewHolder.progressView.setVisibility(View.INVISIBLE);
            viewHolder.orderView.setVisibility(View.GONE);
            }
        else if (editmode == EDIT_MOVE)
            {
            viewHolder.imageView.setImageResource(R.drawable.ic_tobottom);
            viewHolder.imageView2.setVisibility(View.INVISIBLE);
            viewHolder.progressView.setVisibility(View.INVISIBLE);
            viewHolder.orderView.setVisibility(View.GONE);
            }
        else
            {
            viewHolder.imageView2.setVisibility(View.INVISIBLE);
            if (bShowOrder)
                {
                viewHolder.orderView.setVisibility(View.VISIBLE);
                }
            else
                {
                viewHolder.orderView.setVisibility(View.GONE);
                }
            }
     }

    public class AlbumImage
    {
        Bitmap bitmap;
        PlayListModel playListModel;
        MyPlayListAdapter myPlayListAdapter;
        ViewHolder viewHolder;
    }

    public static class PicLoad extends AsyncTask<AlbumImage,Void,AlbumImage>
    {
        @Override
        protected AlbumImage doInBackground(AlbumImage... albumImages)
        {
            AlbumImage albumImage=albumImages[0];

            albumImage.bitmap=Tools.getMediaBitmap(albumImage.playListModel.getFile());
            if (albumImage.bitmap!=null)
                albumImage.bitmap=Bitmap.createScaledBitmap(albumImage.bitmap,albumImage.myPlayListAdapter.widthImage,albumImage.bitmap.getHeight()*albumImage.myPlayListAdapter.widthImage/albumImage.bitmap.getWidth(),true);
            return albumImage;
        }

        @Override
        protected void onPostExecute(AlbumImage result) {
            if (result.viewHolder.url.equals(result.playListModel.getUrl()))
                {
                result.myPlayListAdapter.cache.put(result.playListModel.getUrl(),result.bitmap);
                if (result.bitmap!=null && !result.playListModel.getId().equals(result.myPlayListAdapter.playingid) && result.myPlayListAdapter.editmode == EDIT_PLAY)
                    {
                     result.viewHolder.imageView.setImageBitmap(result.bitmap);
                    }
                }
        }

    }

    @Override
    public long getItemId(int position)
    {
        return Long.parseLong( mDataSet.get(position).getId());
    }

    @Override
    public int getItemCount()
    {
        return mDataSet.size();
    }

    public void setEditmode(int mode)
    {
        editmode = mode;
        notifyDataSetChanged();
    }

    public int getEditmode()
    {
        return editmode;
    }

    public void setPlayingId(String id, int showing)
    {
        if (playingid!=null && playingid.equals(id))
            {
            int pos=mDataSet.getPositon(playingid);
            notifyItemChanged(pos);
            }
        else
            {
            if (playingid!=null)
                {
                int pos=mDataSet.getPositon(playingid);
                notifyItemChanged(pos);
                }
            if (id!=null)
                {
                int pos = mDataSet.getPositon(id);
                notifyItemChanged(pos);
                }
            }
        playingid = id;
        eShowing = showing;
    }

    public interface Controller
    {
        void itemSelected(CustomListItem notificationStyleSelected);
    }
}
