/**
 * Copyright (C) 2014 Casper Jørgensen
 * Modifications copyright (C) 2018 Achim Kaiser
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.kaisersite.mylibrary;

import android.app.Fragment;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Bundle;
import android.os.IBinder;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import android.util.Log;
import android.widget.Toast;

import org.fourthline.cling.android.AndroidUpnpService;
import org.fourthline.cling.android.AndroidUpnpServiceImpl;
import org.fourthline.cling.model.action.ActionException;
import org.fourthline.cling.model.action.ActionInvocation;
import org.fourthline.cling.model.message.UpnpResponse;
import org.fourthline.cling.model.meta.Device;
import org.fourthline.cling.model.meta.LocalDevice;
import org.fourthline.cling.model.meta.RemoteDevice;
import org.fourthline.cling.model.meta.Service;
import org.fourthline.cling.model.types.ErrorCode;
import org.fourthline.cling.registry.DefaultRegistryListener;
import org.fourthline.cling.registry.Registry;
import org.fourthline.cling.support.contentdirectory.callback.Browse;
import org.fourthline.cling.support.model.BrowseFlag;
import org.fourthline.cling.support.model.DIDLContent;
import org.fourthline.cling.support.model.DIDLObject;

import org.fourthline.cling.support.model.SortCriterion;
import org.fourthline.cling.support.model.container.Container;
import org.fourthline.cling.support.model.item.Item;

import java.net.URI;
import java.util.ArrayList;
import java.util.Set;
import java.util.Stack;

public class ContentDirectoryBrowseTaskFragment extends Fragment
{

    private static final String TAG = "ContentFragment";

    public static interface Callbacks
    {
        void onDisplayDevices();

        void onDisplayDirectories(String title);

        void onDisplayItems(ArrayList<ItemModel> items);

        void onDisplayItemsError(String error);

        void onContentDeviceAdded(DeviceModel device);

        void onAVTransportDeviceAdded(DeviceModel device);

        void onDeviceRemoved(DeviceModel device);
    }

    private Callbacks mCallbacks;
    private BrowseRegistryListener mListener = new BrowseRegistryListener();
    private AndroidUpnpService mService;
    private Stack<ItemModel> mFolders = new Stack<ItemModel>();
    private boolean bIsShowingDeviceList = true;
    private DeviceModel mCurrentDevice = null;
    private Context mContext;

    @Override
    public void onAttach(Context context)
    {
        super.onAttach(context);

        if (context == null)
            return;

        mContext = context;
        mCallbacks = (Callbacks) context;
    }

    @Override
    public void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);
        setRetainInstance(true);

        bindServiceConnection();
    }

    @Override
    public void onDetach()
    {
        super.onDetach();
        mCallbacks = null;
    }

    @Override
    public void onDestroy()
    {
        super.onDestroy();

        unbindServiceConnection();
    }

    public void navigateTo(Object model, boolean bBrowseAVTransport)
    {
        if (model instanceof DeviceModel)
            {

            DeviceModel deviceModel = (DeviceModel) model;
            Device device = deviceModel.getDevice();

            if (device.isFullyHydrated())
                {
                if (!bBrowseAVTransport)
                    {
                    Service conDir = deviceModel.getContentDirectory();

                    if (conDir != null)
                        mService.getControlPoint().execute(
                                new CustomContentBrowseActionCallback(conDir, "0"));

                    if (mCallbacks != null)
                        mCallbacks.onDisplayDirectories(deviceModel.getTitle());

                    bIsShowingDeviceList = false;

                    mCurrentDevice = deviceModel;
                    }
                else
                    {
                    CurrentDevice.setDevice(getContext(),device.getIdentity().getUdn().getIdentifierString(),deviceModel.getTitle());

                    Intent intent = new Intent(MyMediaService.CLOSE_PLAYER);
                    getContext().sendBroadcast(intent);
                    Intent bcintent = new Intent(MyMediaPlayerUPNP.UPNP_INFO);
                    LocalBroadcastManager.getInstance(getContext()).sendBroadcast(bcintent);
                    getActivity().finish();
                    }
                }
            else
                {
                Toast.makeText(mContext,R.string.toast_info_still_loading,Toast.LENGTH_SHORT).show();
                }
            }

        if (model instanceof ItemModel)
            {

            ItemModel item = (ItemModel) model;

            if (item.isContainer())
                {
                if (mFolders.isEmpty())
                    mFolders.push(item);
                else if (!mFolders.peek().getId().equals(item.getId()))
                    mFolders.push(item);

                mService.getControlPoint().execute(
                        new CustomContentBrowseActionCallback(item.getService(),
                                item.getId()));
                if (mCallbacks != null)
                    mCallbacks.onDisplayDirectories(item.getTitle());
                }
            else
                {
                DownloadJobService.startDownload(getActivity(),item.getTitle(),item.getDescription(),item.getUrl(),item.getDescription2(),item.getDuration());
                }
            }

        if (model instanceof CustomListItem)
            {
            CustomListItem item = (CustomListItem) model;
            if (item.getTitle().equals(".."))
                {
                goBack();
                }
            }
    }

    public Boolean goBack()
    {
        if (mFolders.empty())
            {
            if (!bIsShowingDeviceList)
                {
                bIsShowingDeviceList = true;
                if (mCallbacks != null)
                    mCallbacks.onDisplayDevices();
                }
            else
                {
                return true;
                }
            }
        else
            {
            ItemModel item = mFolders.pop();

            mService.getControlPoint().execute(
                    new CustomContentBrowseActionCallback(item.getService(),
                            item.getContainer().getParentID()));
            String title;
            if (mFolders.empty())
                title=new DeviceModel(0,item.getService().getDevice()).getTitle();
            else
                title=mFolders.peek().getTitle();

            if (mCallbacks != null)
                mCallbacks.onDisplayDirectories(title);
            }

        return false;
    }

    public void refreshCurrent(boolean bUpdatedDeviceList)
    {
        if (mService == null)
            return;

        if (bIsShowingDeviceList || bUpdatedDeviceList)
            {
            mService.getRegistry().removeAllRemoteDevices();

            for (Device device : mService.getRegistry().getDevices())
                mListener.deviceAdded(device);

            mService.getControlPoint().search();
            }
        if (!bIsShowingDeviceList)
            {
            if (!mFolders.empty())
                {
                ItemModel item = mFolders.peek();
                if (item == null)
                    return;

                mService.getControlPoint().execute(
                        new CustomContentBrowseActionCallback(item.getService(),
                                item.getId()));
                if (mCallbacks != null)
                    mCallbacks.onDisplayDirectories(item.getTitle());

                }
            else
                {
                if (mCurrentDevice != null)
                    {
                    Service service = mCurrentDevice.getContentDirectory();
                    if (service != null)
                        {
                        mService.getControlPoint().execute(
                                new CustomContentBrowseActionCallback(service, "0"));
                        String title=new DeviceModel(0,service.getDevice()).getTitle();
                        if (mCallbacks != null)
                            mCallbacks.onDisplayDirectories(title);
                        }
                    }
                }
            }
    }

    private Boolean bindServiceConnection()
    {
        Context context = mContext.getApplicationContext();
        if (context == null)
            return false;

        context.bindService(
                new Intent(mContext, AndroidUpnpServiceImpl.class),
                serviceConnection, Context.BIND_AUTO_CREATE
        );

        Tools.clientConnected();
        return true;
    }

    private Boolean unbindServiceConnection()
    {
        if (mService != null)
            mService.getRegistry().removeListener(mListener);

        final Context context = mContext.getApplicationContext();
        if (context == null)
            return false;

        if (Tools.clientDisconnected()>0)
            return true;
        Log.d(TAG,"UPNP Service stop");
        new Thread()
        {
            public void run()
            {
                mService = null;

                Set<Thread> threadSet = Thread.getAllStackTraces().keySet();

                for (Thread thread : threadSet)
                    {
                    if (thread.getName().startsWith("cling"))
                        {
                        thread.interrupt();
                        }
                    }

                try
                    {
                    context.unbindService(serviceConnection);
                    }
                catch (IllegalArgumentException ignore)
                    {
                    } // Already unbound
            }
        }.start();

        return true;
    }

    private ServiceConnection serviceConnection = new ServiceConnection()
    {
        public void onServiceConnected(ComponentName className, IBinder service)
        {
            mService = (AndroidUpnpService) service;
            mService.getRegistry().addListener(mListener);

            for (Device device : mService.getRegistry().getDevices())
                mListener.deviceAdded(device);

            mService.getControlPoint().search();
        }

        public void onServiceDisconnected(ComponentName className)
        {
            mService = null;
        }
    };

    private class BrowseRegistryListener extends DefaultRegistryListener
    {
        @Override
        public void remoteDeviceDiscoveryStarted(Registry registry, RemoteDevice device)
        {
            deviceAdded(device);
        }

        @Override
        public void remoteDeviceDiscoveryFailed(Registry registry, RemoteDevice device, Exception ex)
        {
            deviceRemoved(device);
        }

        @Override
        public void remoteDeviceAdded(Registry registry, RemoteDevice device)
        {
            deviceAdded(device);
        }

        @Override
        public void remoteDeviceRemoved(Registry registry, RemoteDevice device)
        {
            deviceRemoved(device);
        }

        @Override
        public void localDeviceAdded(Registry registry, LocalDevice device)
        {
            deviceAdded(device);
        }

        @Override
        public void localDeviceRemoved(Registry registry, LocalDevice device)
        {
            deviceRemoved(device);
        }

        public void deviceAdded(Device device)
        {

            DeviceModel deviceModel = new DeviceModel(R.drawable.ic_device, device);

            Service conDir = deviceModel.getContentDirectory();
            if (conDir != null)
                {
                if (mCallbacks != null)
                    mCallbacks.onContentDeviceAdded(deviceModel);
                }

            Service traDir = deviceModel.getAVTransport();
            if (traDir != null)
                {
                if (mCallbacks != null)
                    mCallbacks.onAVTransportDeviceAdded(deviceModel);
                }
        }

        public void deviceRemoved(Device device)
        {
            if (mCallbacks != null)
                mCallbacks.onDeviceRemoved(new DeviceModel(R.drawable.ic_device, device));
        }
    }

    private class CustomContentBrowseActionCallback extends Browse
    {
        private Service service;

        public CustomContentBrowseActionCallback(Service service, String id)
        {
            super(service, id, BrowseFlag.DIRECT_CHILDREN, "*", 0, 99999l,
                    (SortCriterion[])null/*new SortCriterion(true, "dc:title")*/); //doesn't work with some servers

            this.service = service;
        }

        private ItemModel createItemModel(DIDLObject item)
        {

            ItemModel itemModel = new ItemModel(getResources(),
                    R.drawable.ic_folder, service, item);

            URI albumIcon = item.getFirstPropertyValue(DIDLObject.Property.UPNP.ALBUM_ART_URI.class);
            if (albumIcon != null)
                itemModel.setIconUrl(albumIcon.toString());

            URI usableIcon = item.getFirstPropertyValue(DIDLObject.Property.UPNP.ICON.class);
            if (albumIcon==null && usableIcon != null)
                itemModel.setIconUrl(usableIcon.toString());

            if (item instanceof Item)
                {
                itemModel.setIcon(R.drawable.ic_load);
                }

            return itemModel;
        }

        @Override
        public void received(final ActionInvocation actionInvocation, final DIDLContent didl)
        {

            ArrayList<ItemModel> items = new ArrayList<ItemModel>();

            try
                {
                for (Container childContainer : didl.getContainers())
                    items.add(createItemModel(childContainer));

                for (Item childItem : didl.getItems())
                    {
                    // Only audio but no broadcasts
                    if (childItem.getClazz().getValue().startsWith("object.item.audioItem") &&
                            !childItem.getClazz().getValue().equals("object.item.audioItem.audioBroadcast"))
                        items.add(createItemModel(childItem));
                    }

                if (mCallbacks != null)
                    mCallbacks.onDisplayItems(items);

                }
            catch (Exception ex)
                {
                actionInvocation.setFailure(new ActionException(
                        ErrorCode.ACTION_FAILED,
                        "Can't create list childs: " + ex, ex));
                failure(actionInvocation, null, ex.getMessage());
                }
        }

        @Override
        public void updateStatus(Status status)
        {

        }

        @Override
        public void failure(ActionInvocation invocation, UpnpResponse response, String s)
        {
            if (mCallbacks != null)
                mCallbacks.onDisplayItemsError(createDefaultFailureMessage(invocation, response));
        }
    }

    private class CustomContentBrowseTestCallback extends Browse
    {
        private Device device;
        private Service service;

        public CustomContentBrowseTestCallback(Device device, Service service)
        {
            super(service, "0", BrowseFlag.DIRECT_CHILDREN, "*", 0, 99999l,
                    (SortCriterion[])null/*new SortCriterion(true, "dc:title")*/); // doesn't work with some servers

            this.device = device;
            this.service = service;
        }

        @Override
        public void received(final ActionInvocation actionInvocation, final DIDLContent didl)
        {
            if (mCallbacks != null)
                mCallbacks.onContentDeviceAdded(new DeviceModel(R.drawable.ic_device, device));
        }

        @Override
        public void updateStatus(Status status)
        {

        }

        @Override
        public void failure(ActionInvocation actionInvocation, UpnpResponse upnpResponse, String s)
        {

        }
    }
}